/*---------------------------------------------------------------------------------
Name               : config.c
Author             : Marvin Raaijmakers
Description        : Asks the user for information and adds it to the XML document.
Date of last change: 18-Jan-2006
History            :
                     18-Jan-2006 - Modified source so that it can also handle ACPI
                                   hotkeys
                                 - Added keycode_in_use[] array to get_key() to
                                   to avoid multiple use of a keycode

    Copyright (C) 2005-2006 Marvin Raaijmakers

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or any later version.
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

    You can contact me at: marvinr(at)users(dot)sf(dot)net
    (replace (at) by @ and (dot) by .)
-----------------------------------------------------------------------------------*/
#include <linux/input.h>

#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <errno.h>

#include <mxml/mxml.h>
#include <keytouch-editor.h>
#include <config_element_types.h>
#include <string_to_keycode/string_to_keycode.h>


void
xml_add_string_element (	XmlContent	*parent,
				XmlElementType	*element_type,
				char		*string        )
/*
Input:
	element_type	- The type of the new element
	string		- The string the new element will contain
Output:
	parent		- The parent element where the new element will be add to
Returns:
	-
Description:
	This functions creates a new element of type element_type and adds it to parent.
	The new element will contain the string 'string'.
*/
{
	XmlContent	*new_element,
			*xml_string;
	
	new_element = XmlCreateElement(element_type, FALSE);
	XmlContentAppend (new_element, XmlElementContentList(parent));
	xml_string = XmlCreateContent(XmlCType_String);
	XmlStringSetString (xml_string, string);
	XmlContentAppend (xml_string, XmlElementContentList(new_element));
}


void
get_string (	char	*string,
		int	max_len    )
/*
Input:
	max_len	- The maximum number of characters minus one to read.
Output:
	string	- The readen string
Returns:
	-
Description:
	This function reads in at most size characters from stdin and stores them
	into the buffer pointed to by string. Reading stops after an EOF or a
	newline, which will not be included. A '\0' is stored after the last
	character in the buffer.
*/
{
	char *new_line;
	
	fgets (string, max_len+1, stdin);
	new_line = strchr(string, '\n');
	/* If the line was not completely read */
	if (new_line == NULL)
	{
		/* Read stdin until the end of line */
		while (getchar() != '\n')
			; /* NULL Statement */
	}
	else
	{
		*new_line = '\0';
	}
}


Boolean
get_key (	char			*device_name,
		int			acpi_socket,
		Boolean			is_i8042,
		Boolean			is_acpi,
		XmlContent		*key_list_element,
		XmlElementType		*key_etype,
		XmlElementType		*scancode_eventdescr_etype,
		XmlElementType		*keycode_etype,
		XmlElementType		*key_name_etype,
		XmlElementType		*default_action_etype,
		XmlElementType		*plugin_name_etype,
		XmlElementType		*plugin_function_etype,
		XmlAttributeName	*action_type_attrname,
		XmlAttributeName	*key_type_attrname     )
/*
Input:
	device_name	- The file name of the event device to read from when is_acpi
			  is FALSE
	acpi_socket	- If is_acpi is TRUE the ACPI event descriptions will be read
			  from this socket
	is_i8042	- Indicates if the keyboard is connected to an i8042
			  controler
	is_acpi		- Indicates if this function should read scancodes or event
			  descriptions
	*_etype		- Element types this functions uses to add a key to
			  the key list.
Output:
	key_list_element	- The key will be added to this list.
Returns:
	TRUE if a key was added, otherwise FALSE.
Description:
	This function creates asks the user to press an extra function key. After
	that the user will be asked for the keys name, keycode and its default
	action. An XmlElement of type key_etype will be created containing this
	information. The XmlElement will be added to key_list_element.
	If is_i8041 is TRUE the user can cancel all this by pressing the return key
	instead of an extra function key. If is_i8042 is FALSE the user will be
	asked if he wants to cancel or continue before pressing the extra function
	key. When canceling no key will be added to key_list_element.
*/
{
	int		event_device;
	unsigned int	scancode,
			keycode;
	char		scancode_string[MAX_SCANCODE_LEN+1],
			key_name[MAX_KEY_NAME_LEN+1],
			keycode_string[MAX_KEYCODE_LEN+1],
			default_program[MAX_PROGRAM_LEN+1],
			plugin_name[MAX_PLUGIN_NAME_LEN+1],
			plugin_function[MAX_PLUGIN_FUNCTION_LEN+1],
			*eventdescr,
			c;
	XmlContent	*key_element,
			*default_action_element;
	Boolean		add_key;
	/* The array below is used to avoid multiple use of a keycode */
	static Boolean	keycode_in_use[NUM_KEYCODES] = {[0 ... NUM_KEYCODES-1] = FALSE};
	
	if (!is_acpi)
	{
		/* If opening the event device failed */
		if ((event_device = open(device_name, O_RDONLY)) < 0)
		{
			perror("keytouch-editor");
			exit (1);
		}
	}
	
	add_key = TRUE;
	if (is_i8042)
	{
		printf ("\nPress an extra function key or press enter to finish...\n");
	}
	else
	{
		printf ("\nPress return to a new key or enter F followed by return to finish... ");
		c = getchar();
		if (c == 'F')
		{
			add_key = FALSE;
		}
		else
		{
			if (c != '\n')
			{
				/* Read stdin until the end of line */
				while (getchar() != '\n')
					; /* NULL Statement */
			}
			printf ("Press an extra function...\n");
		}
	}
	if (add_key)
	{
		if (is_acpi)
		{
			eventdescr = read_eventdescr(acpi_socket);
			if (!eventdescr)
			{
				add_key = FALSE;
				if (errno == EPIPE)
				{
					fputs ("Error: events file connection closed", stderr);
					exit (EXIT_FAILURE);
				}
			}
			else
			{
				add_key = TRUE;
			}
		}
		else
		{
			scancode = get_scancode(event_device);
			add_key = (scancode != 0);
		}
	}
	if (add_key)
	{
		/* Create and append the key element */
		key_element = XmlCreateElement(key_etype, FALSE);
		XmlContentAppend (key_element, XmlElementContentList(key_list_element));
		if (is_acpi)
		{
			XmlSetAttributeValue (key_type_attrname, ATTR_VAL_ACPIHOTKEY, key_element);
		}
		
		printf ("The name of the key you have pressed: ");
		get_string (key_name, MAX_KEY_NAME_LEN);
		xml_add_string_element (key_element, key_name_etype, key_name);
		
		
		if (is_acpi)
		{
			/* Add the event-descr element to the key element */
			xml_add_string_element (key_element, scancode_eventdescr_etype, eventdescr);
		}
		else
		{
			/* Convert the scancode to a string */
			snprintf (scancode_string, MAX_SCANCODE_LEN+1, "%d", scancode);
			/* Add the scancode element to the key element */
			xml_add_string_element (key_element, scancode_eventdescr_etype, scancode_string);
		}
		
		printf ("An appropriate keycode for the key: ");
		/* While the keycode is invalid or was already chosen before */
		while (1)
		{
			get_string (keycode_string, MAX_KEYCODE_LEN);
			keycode = string_to_kernel_keycode(keycode_string);
			if (!keycode)
			{
				printf ("\"%s\" is not a valid keycode. Please reenter: ", keycode_string);
			}
			else if (keycode_in_use[keycode])
			{
				printf ("You cannot use keycode \"%s\" twice. Please reenter: ", keycode_string);
			}
			else
			{
				break;
			}
		}
		keycode_in_use[keycode] = TRUE;
		xml_add_string_element (key_element, keycode_etype, keycode_string);
		
		printf ("You will now have to fill in the default action for this key.\n"
			"This can be a program or a plugin. Just enter the name of the\n"
			"or enter \"plugin\" to choose a plugin: ");
		get_string (default_program, MAX_PROGRAM_LEN);
		if (strcmp(default_program, "plugin") == EQUAL)
		{
			default_action_element = XmlCreateElement(default_action_etype, FALSE);
			XmlSetAttributeValue (action_type_attrname, ATTR_VAL_PLUGIN, default_action_element);
			XmlContentAppend (default_action_element, XmlElementContentList(key_element));
			
			printf (" The name of the plugin: ");
			get_string (plugin_name, MAX_PLUGIN_NAME_LEN);
			xml_add_string_element (default_action_element, plugin_name_etype, plugin_name);
			
			printf (" The function of the plugin: ");
			get_string (plugin_function, MAX_PLUGIN_FUNCTION_LEN);
			xml_add_string_element (default_action_element, plugin_function_etype, plugin_function);
		}
		else
		{
			xml_add_string_element (key_element, default_action_etype, default_program);
		}
	}
	if (!is_acpi)
	{
		close (event_device);
	}
	return (add_key);
}


void
get_key_list (	XmlContent	*keyboard_element,
		XmlDocument	*document,
		char		*device_name,
		int		acpi_socket,
		Boolean		is_i8042,
		Boolean		is_acpi  )
/*
Input:
	device_name	- The file name of the event device to read from when is_acpi
			  is FALSE
	acpi_socket	- If is_acpi is TRUE the ACPI event descriptions will be read
			  from this socket
	is_i8042	- Indicates if the keyboard is connected to an i8042
			  controler
	is_acpi		- Indicates if this function should read scancodes or event
			  descriptions
Output:
	keyboard_element	- The key list will be added to this element
	document		- The document containing keyboard_element
				  and after calling this function the key list.
Returns:
	-
Description:
	This function creates a key list that will be added to keyboard_element. It
	uses get_key() to retrieve information about the keys.
*/
{
	XmlContent		*key_list_element;
	XmlElementType		*key_list_etype,
				*key_etype,
				*scancode_eventdescr_etype,
				*keycode_etype,
				*key_name_etype,
				*default_action_etype,
				*plugin_name_etype,
				*plugin_function_etype;
	XmlAttributeName	*action_type_attrname,
				*key_type_attrname;
	
	key_list_etype = XmlGetElementType(ELEM_TYPE_KEYLIST, document, TRUE);
	key_list_element = XmlCreateElement(key_list_etype, FALSE);
	XmlContentAppend (key_list_element, XmlElementContentList(keyboard_element));
	
	key_list_etype = XmlGetElementType(ELEM_TYPE_KEYLIST, document, TRUE);
	key_etype = XmlGetElementType(ELEM_TYPE_KEY, document, TRUE);
	if (is_acpi)
	{
		scancode_eventdescr_etype = XmlGetElementType(ELEM_TYPE_EVENTDESCR, document, TRUE);
	}
	else
	{
		scancode_eventdescr_etype = XmlGetElementType(ELEM_TYPE_SCANCODE, document, TRUE);
	}
	keycode_etype = XmlGetElementType(ELEM_TYPE_KEYCODE, document, TRUE);
	key_name_etype = XmlGetElementType(ELEM_TYPE_KEYNAME, document, TRUE);
	default_action_etype = XmlGetElementType(ELEM_TYPE_DEFAULTACTION, document, TRUE);
	plugin_name_etype = XmlGetElementType(ELEM_TYPE_PLUGINNAME, document, TRUE);
	plugin_function_etype = XmlGetElementType(ELEM_TYPE_PLUGINFUNCTION, document, TRUE);
	
	action_type_attrname = XmlGetAttributeName(ATTR_NAME_ACTIONTYPE, document, TRUE);
	key_type_attrname = XmlGetAttributeName(ATTR_NAME_KEYTYPE, document, TRUE);
	
	while (get_key(device_name, acpi_socket, is_i8042, is_acpi, key_list_element,
	               key_etype, scancode_eventdescr_etype, keycode_etype,
	               key_name_etype, default_action_etype,
	               plugin_name_etype, plugin_function_etype, 
	               action_type_attrname, key_type_attrname) )
		; /* NULL Statement */
}


void
get_keyboard_name (	XmlContent	*keyboard_element,
			XmlDocument	*document )
/*
Input:
	-
Output:
	keyboard_element	- The keyboard-info element will be add to this
				  element.
	document		- The document that contains keyboard_element. New
				  element types and attribute names will be add to
				  this document.
Returns:
	-
Description:
	This functions creates a keyboard-info element and adds it to
	keyboard_element. The keyboard-info element contains a keyboard-name sub-
	element which contains the following sub-elements:
	- manufacturer: The name of the keyboard's manufacturer, filled in by the
	  user.
	- model: The keyboard model, filled in by the user.
*/
{
	XmlContent	*keyboard_info_element,
			*keyboard_name_element;
	XmlElementType	*element_type;
	char		manufacturer[MAX_MANUFACTURER_LEN+1],
			model[MAX_MODEL_LEN+1];
	
	element_type = XmlGetElementType(ELEM_TYPE_KEYBOARDINFO, document, TRUE);
	keyboard_info_element = XmlCreateElement(element_type, FALSE);
	XmlContentAppend (keyboard_info_element, XmlElementContentList(keyboard_element));
	element_type = XmlGetElementType(ELEM_TYPE_KEYBOARDNAME, document, TRUE);
	keyboard_name_element = XmlCreateElement(element_type, FALSE);
	XmlContentAppend (keyboard_name_element, XmlElementContentList(keyboard_info_element));
	
	printf ("The name of your keyboards manufacturer: ");
	get_string (manufacturer, MAX_MANUFACTURER_LEN);
	printf ("The model of you keyboard: ");
	get_string (model, MAX_MODEL_LEN);
	
	xml_add_string_element (keyboard_name_element,
				XmlGetElementType(ELEM_TYPE_MANUFACTURER, document, TRUE),
				manufacturer);
	xml_add_string_element (keyboard_name_element,
				XmlGetElementType(ELEM_TYPE_MODEL, document, TRUE),
				model);
} 


void
get_file_info (	XmlContent	*keyboard_element,
		XmlDocument	*document          )
/*
Input:
	-
Output:
	keyboard_element	- The file-info element will be add to this element.
	document		- The document that contains keyboard_element. New
				  element types and attribute names will be add to
				  this document.
Returns:
	-
Description:
	This functions creates a file-info element and adds it to keyboard_element.
	The file-info element will contain the following sub-elements:
	- syntax-version: Contains the SYNTAX_VERSION.
	- last-change: Contains todays date.
	- author: The name of the author, filled in by the user.
*/
{
	XmlContent	*file_info_element,
			*last_change_element,
			*last_change_string;
	XmlElementType	*element_type;
	char		author[MAX_AUTHOR_LEN+1],
			date[MAX_DATE_LEN+1];
	time_t		current_time;
	
	element_type = XmlGetElementType(ELEM_TYPE_FILEINFO, document, TRUE);
	file_info_element = XmlCreateElement(element_type, FALSE);
	XmlContentAppend (file_info_element, XmlElementContentList(keyboard_element));
	
	/** Create the syntax-version element containing SYNTAX_VERSION and add it to
	 ** the file info element  **/
	xml_add_string_element (file_info_element,
				XmlGetElementType(ELEM_TYPE_SYNTAXVERSION, document, TRUE),
				SYNTAX_VERSION);
	
	/** Create the last change element and add it to the file info element **/
	element_type = XmlGetElementType(ELEM_TYPE_LASTCHANGE, document, TRUE);
	last_change_element = XmlCreateElement(element_type, FALSE);
	XmlContentAppend (last_change_element, XmlElementContentList(file_info_element));
	/** Get the current time, convert it to a string and put it in date **/
	time (&current_time);
	strftime (date, MAX_DATE_LEN, DATE_FORMAT, localtime(&current_time));
	/* Create the XmlContent string that will contain date */
	last_change_string = XmlCreateContent(XmlCType_String);
	/* Let last_change_string contain date */
	XmlStringSetString (last_change_string, date);
	/* Add last_change_string to last_change_element */
	XmlContentAppend (last_change_string, XmlElementContentList(last_change_element));
	/* Set the date format we used */
	XmlSetAttributeValue (	XmlGetAttributeName (ATTR_NAME_FORMAT, document, TRUE),
				DATE_FORMAT,
				last_change_element );
	
	/** Get the name of the author and create a author element, in the file info
	 ** element, containing this name. */
	printf ("Your (real) name: ");
	get_string (author, MAX_AUTHOR_LEN);
	xml_add_string_element (file_info_element,
				XmlGetElementType(ELEM_TYPE_AUTHOR, document, TRUE),
				author);
}
